package br.com.softilux.softiluxapp.ui.login;

import android.arch.lifecycle.LiveData;
import android.arch.lifecycle.MutableLiveData;

import com.google.gson.Gson;
import com.jakewharton.retrofit2.adapter.rxjava2.HttpException;

import org.apache.commons.lang3.StringUtils;

import java.io.IOException;

import javax.inject.Inject;

import br.com.softilux.core.utils.LoggerUtils;
import br.com.softilux.core.utils.constants.HttpStatus;
import br.com.softilux.core.webservice.ApiResponse;
import br.com.softilux.core.webservice.ResponseTransfer;
import br.com.softilux.softiluxapp.data.repository.LoginRepository;
import br.com.softilux.softiluxapp.transfer.datatransfer.UserResponse;
import io.reactivex.disposables.CompositeDisposable;
import okhttp3.ResponseBody;

/**
 * Created by Vinicius on 14,Abril,2019
 */
public class LoginViewModel extends BaseViewModel {

    private final String TAG = getClass().getSimpleName();
    private final LoginRepository loginRepository;
    private MutableLiveData<ResponseTransfer<UserResponse>> data = new MutableLiveData<>();
    private MutableLiveData<CallStatus> callStatus = new MutableLiveData<>();
    private MutableLiveData<String> callMessage = new MutableLiveData<>();
    private final MutableLiveData<ApiResponse<UserResponse>> progressLogin = new MutableLiveData<>();
    //private Call<ResponseTransfer<UserResponse>> callService;

    public LiveData<ApiResponse<UserResponse>> getProgressLogin() {
        return progressLogin;
    }

    @Inject
    public LoginViewModel(LoginRepository loginRepository) {
        this.loginRepository = loginRepository;
    }

    public void loginWebService(LoginDTO loginDTO) {
            loginRepository.getCompositeDisposable()
                    .add(loginRepository.getAuthService().authUserRx(loginDTO.getEmail(), loginDTO.getPassword())
                    .subscribeOn(loginRepository.getSchedulerProvider().io())
                    .observeOn(loginRepository.getSchedulerProvider().ui())
                    .doOnSubscribe((d) -> progressLogin.setValue(ApiResponse.loading()))
                    .subscribe(
                        result -> onSuccessResult(result),
                        throwable -> onError(throwable)
                    ));



//        progressLogin.setValue(ApiResponse.loading());
//        new Handler(Looper.getMainLooper()).postDelayed(new Runnable() {
//            @Override
//            public void run() {
//                callService = loginRepository.getAuthService().authUser(loginDTO.getEmail(), loginDTO.getPassword());
//                callService.enqueue(new Callback<ResponseTransfer<UserResponse>>() {
//                    @Override
//                    public void onResponse(@NotNull Call<ResponseTransfer<UserResponse>> call, @NotNull Response<ResponseTransfer<UserResponse>> response) {
//                        if (response.isSuccessful()) {
//                            ResponseTransfer<UserResponse> responseAuth = response.body();
//                            loginRepository.salvarUsuario(responseAuth);
//                            progressLogin.setValue(ApiResponse.success(responseAuth));
//                        } else {
//                            handlerCallbackErrorResult2(response.code(), response.errorBody());
//                        }
//                        callService = null;
//                    }
//                    @Override
//                    public void onFailure(@NotNull Call<ResponseTransfer<UserResponse>> call, @NotNull Throwable t) {
//                        progressLogin.setValue(ApiResponse.error(t));
//                        LoggerUtils.error(t, TAG + " loginWebService()");
//                        callService = null;
//                    }
//                });
//            }
//        },2000);
    }

    private void onSuccessResult(ResponseTransfer<UserResponse> response) {
        loginRepository.salvarUsuario(response);
        progressLogin.setValue(ApiResponse.success(response));
    }

    private void onError(Throwable throwable) {
        if (throwable instanceof HttpException) {
            HttpException error = (HttpException) throwable;
            handlerCallbackErrorResult2(error.response().code(), error.response().errorBody());
        } else {
            progressLogin.setValue(ApiResponse.error(throwable));
            LoggerUtils.error(throwable, TAG + " loginWebService()");
        }
    }

    private void handlerCallbackErrorResult2(int code, ResponseBody errorBody) {
        ResponseTransfer responseTransfer = convertErrorBody(errorBody);
        if (responseTransfer != null && StringUtils.isNotBlank(responseTransfer.getMessage())) {
            progressLogin.setValue(ApiResponse.errorBody(responseTransfer));
        } else {
            progressLogin.setValue(ApiResponse.error());
        }
    }

    private void handlerCallbackErrorResult(int code, ResponseBody errorBody) {
        ResponseTransfer responseTransfer = convertErrorBody(errorBody);
        if (responseTransfer != null && StringUtils.isNotBlank(responseTransfer.getMessage())) {
            handlerMessageError(responseTransfer.getMessage());
        } else {
            handlerCallbackErrorResult(code);
        }
    }

    private ResponseTransfer convertErrorBody(ResponseBody errorBody) {
        if ( errorBody == null ) return null;

        ResponseTransfer responseTransfer = null;
        try {
            responseTransfer = loginRepository.getGson().fromJson(errorBody.string(), ResponseTransfer.class);
        } catch (IOException e) {
            LoggerUtils.error(e, TAG + " (convertErrorBody)");
        }
        return responseTransfer;
    }

    private void handlerCallbackErrorResult(int code) {
        HttpStatus httpStatus = HttpStatus.valueOf(code);
        switch (httpStatus) {
            case UNAUTHORIZED:
                handlerMessageError("Login ou senha inválidos");
            case NOT_FOUND:
            case FORBIDDEN:
            case BAD_REQUEST:
            case INTERNAL_SERVER_ERROR:
                handlerMessageError("Erro no servidor! Por favor contate o Administador.");
        }
    }

    private void handlerMessageError(String message) {
        callMessage.setValue(message);
    }

    private void handlerSuccessData(ResponseTransfer<UserResponse> response) {
        data.setValue(response);
    }

    private void updateCallableStatus(CallStatus event) {
        callStatus.setValue(event);
    }

    public LiveData<ResponseTransfer<UserResponse>> getData() {
        return data;
    }

    public LiveData<CallStatus> getCallStatus() {
        return callStatus;
    }

    public LiveData<String> getCallMessage() {
        return callMessage;
    }

    @Override
    protected void onCleared() {
        loginRepository.getCompositeDisposable().dispose();
        super.onCleared();
    }
}
