package br.com.softilux.softiluxapp.service;

import android.Manifest;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.location.Address;
import android.location.Location;
import android.os.Bundle;
import android.os.IBinder;
import android.support.annotation.Nullable;
import android.support.v4.app.ActivityCompat;

import com.blankj.utilcode.util.NetworkUtils;
import com.blankj.utilcode.util.TimeUtils;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;

import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;

import javax.inject.Inject;

import br.com.softilux.core.utils.GeocoderUtils;
import br.com.softilux.core.utils.LoggerUtils;
import br.com.softilux.core.webservice.ResponseTransfer;
import br.com.softilux.softiluxapp.SoftiluxApp;
import br.com.softilux.softiluxapp.data.dao.AtendimentoDAO;
import br.com.softilux.softiluxapp.data.dao.TimelineDAO;
import br.com.softilux.softiluxapp.data.db.model.AtendimentoModel;
import br.com.softilux.softiluxapp.data.db.model.CurrentChamadoAtendimento;
import br.com.softilux.softiluxapp.data.db.model.LocationCapture;
import br.com.softilux.softiluxapp.data.db.model.TimelineModel;
import br.com.softilux.softiluxapp.data.db.model.TipoInclusaoTimeLine;
import br.com.softilux.softiluxapp.data.nosql.IPaperBook;
import br.com.softilux.softiluxapp.di.component2.ApplicationComponent;
import br.com.softilux.softiluxapp.di.component2.ServiceComponent;
import br.com.softilux.softiluxapp.di.component2.ServiceModule;
import br.com.softilux.softiluxapp.transfer.datatransfer.CadastrarNovaTimelineNetworkingRequest;
import br.com.softilux.softiluxapp.transfer.services.IAtendimentoService;
import br.com.softilux.softiluxapp.transfer.services.ITimelineService;
import br.com.softilux.softiluxapp.transfer.services.TimelineRequestResponse;
import br.com.softilux.softiluxapp.utils.rx.SchedulerProvider;
import io.reactivex.disposables.CompositeDisposable;


/**
 * Created by devdeeds.com on 27-09-2017.
 */

public class LocationMonitoringService extends Service implements
        GoogleApiClient.ConnectionCallbacks, GoogleApiClient.OnConnectionFailedListener,
        LocationListener {

    private static final String TAG = LocationMonitoringService.class.getSimpleName();

    @Inject
    IAtendimentoService mAtendimentoService;

    @Inject
    ITimelineService iTimelineService;

    @Inject
    TimelineDAO mTimelineDAO;

    @Inject
    AtendimentoDAO mAtendimentoDAO;

    @Inject
    CompositeDisposable compositeDisposable;

    @Inject
    SchedulerProvider schedulerProvider;

    GoogleApiClient mLocationClient;

    @Inject
    IPaperBook iPaperBook;

    LocationRequest mLocationRequest = new LocationRequest();

    ServiceComponent serviceComponent;

    public static final String ACTION_LOCATION_BROADCAST = LocationMonitoringService.class.getName() + "LocationBroadcast";
    public static final String EXTRA_LATITUDE = "extra_latitude";
    public static final String EXTRA_LONGITUDE = "extra_longitude";

    @Override
    public void onCreate() {
        super.onCreate();
        serviceComponent = getApplicationComponent()
                .newServiceComponent(new ServiceModule(this));
        getServiceComponent().inject(this);
    }

    private ApplicationComponent getApplicationComponent() {
        return ((SoftiluxApp) getApplication()).getComponent();
    }

    public ServiceComponent getServiceComponent() {
        return serviceComponent;
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        mLocationClient = new GoogleApiClient.Builder(this)
                .addConnectionCallbacks(this)
                .addOnConnectionFailedListener(this)
                .addApi(LocationServices.API)
                .build();

        mLocationRequest.setInterval(360000);
        mLocationRequest.setFastestInterval(360000);

        int priority = LocationRequest.PRIORITY_HIGH_ACCURACY; //by default
        //PRIORITY_BALANCED_POWER_ACCURACY, PRIORITY_LOW_POWER, PRIORITY_NO_POWER are the other priority modes


        mLocationRequest.setPriority(priority);
        mLocationClient.connect();

        //Make it stick to the notification panel so it is less prone to get cancelled by the Operating System.
        return START_STICKY;
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    /*
     * LOCATION CALLBACKS
     */
    @Override
    public void onConnected(Bundle dataBundle) {
        if (ActivityCompat.checkSelfPermission(this, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED && ActivityCompat.checkSelfPermission(this, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            // TODO: Consider calling
            //    ActivityCompat#requestPermissions
            // here to request the missing permissions, and then overriding
            //   public void onRequestPermissionsResult(int requestCode, String[] permissions,
            //                                          int[] grantResults)
            // to handle the case where the user grants the permission. See the documentation
            // for ActivityCompat#requestPermissions for more details.

            LoggerUtils.d(TAG + " == Error On onConnected() Permission not granted");
            //Permission not granted by user so cancel the further execution.

            return;
        }
        LocationServices.FusedLocationApi.requestLocationUpdates(mLocationClient, mLocationRequest, this);

        LoggerUtils.d(TAG + " Connected to Google API");
    }

    /*
     * Called by Location Services if the connection to the
     * location client drops because of an error.
     */
    @Override
    public void onConnectionSuspended(int i) {
        LoggerUtils.d(TAG + " Connection suspended");
    }

    //to get the location change
    @Override
    public void onLocationChanged(Location location) {
        LoggerUtils.d(TAG, "Location changed");

        if (location != null) {
            LoggerUtils.d(TAG, "== location != null");

            //Send result to activities
            sendMessageToUI(String.valueOf(location.getLatitude()), String.valueOf(location.getLongitude()));
        }
    }

    private void sendMessageToUI(String lat, String lng) {
        if ( !NetworkUtils.isConnected() ) {
            LoggerUtils.d(TAG + " Internet turn off");
            return;
        }

        if (StringUtils.isNotBlank(lat) && StringUtils.isNotBlank(lng) && iPaperBook.getCurrentAtendimento() != null) {
            AtendimentoModel atendimentoModel = iPaperBook.getCurrentAtendimento().getAtendimentoModel();
            Address address = getAddress(this, Double.valueOf(lat), Double.valueOf(lng));
            LoggerUtils.info("LocationMonitoringService " + lat + lng);
            CadastrarNovaTimelineNetworkingRequest request = new CadastrarNovaTimelineNetworkingRequest();
            // ------------------------------ Preencher dados da nova timeline
            request.gerarNovaTimelineAssincrona(atendimentoModel);
            // ------------------------------ Preencher localizacao
            LocationCapture locationCapture = new LocationCapture(address);
            LoggerUtils.info(locationCapture.toString());
            request.setDataLocationCapture(locationCapture);

            compositeDisposable.add(iTimelineService.salvarNova(request)
                    .observeOn(schedulerProvider.ui())
                    .subscribeOn(schedulerProvider.io())
                    .subscribe(result -> cadastrarNovaTimeline(result, request),
                            throwable -> {
                                LoggerUtils.error(throwable, this.getClass().getSimpleName());
                            })
            );

//        Log.d(TAG, "Sending info...");
//
//        Intent intent = new Intent(ACTION_LOCATION_BROADCAST);
//        intent.putExtra(EXTRA_LATITUDE, lat);
//        intent.putExtra(EXTRA_LONGITUDE, lng);
//        LocalBroadcastManager.getInstance(this).sendBroadcast(intent);
        }
    }

    private void cadastrarNovaTimeline(ResponseTransfer<TimelineRequestResponse> result, CadastrarNovaTimelineNetworkingRequest request) {
        if (result.hasData()) {
            CurrentChamadoAtendimento currentAtendimento = iPaperBook.getCurrentAtendimento();
            AtendimentoModel atendimentoModel = currentAtendimento.getAtendimentoModel();
            TimelineModel timelineModel = TimelineModel.cadastrarNovaTimeline(result.getData(), request, atendimentoModel);
            long insert = mTimelineDAO.insert(timelineModel);
            TimelineModel entity = mTimelineDAO.findByIdApp(insert);
            atendimentoModel.getTimelines().add(entity);
            iPaperBook.salvarCurrentAtendimento(currentAtendimento);
            mAtendimentoDAO.update(atendimentoModel);
        }
    }

    public Address getAddress(Context context, double latitude, double longitude)  {
        return GeocoderUtils.getAddress(latitude, longitude, context);
    }

    @Override
    public void onConnectionFailed(@NotNull ConnectionResult connectionResult) {
        LoggerUtils.d(TAG +  " Failed to connect to Google API");
    }

    @Override
    public void onDestroy() {
        LoggerUtils.info("Service destroyed");
        compositeDisposable.dispose();
        super.onDestroy();
    }
}
