package br.com.softilux.softiluxapp.data.repository;

import android.support.annotation.NonNull;

import com.google.gson.Gson;

import org.apache.commons.collections4.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.stream.Collectors;

import javax.inject.Inject;

import br.com.softilux.core.dao.QueryBuilder;
import br.com.softilux.core.webservice.ResponseListTransfer;
import br.com.softilux.core.webservice.ResponseTransfer;
import br.com.softilux.softiluxapp.data.dao.ChamadoDAO;
import br.com.softilux.softiluxapp.data.db.model.ChamadoFilterModel;
import br.com.softilux.softiluxapp.data.db.model.ChamadoModel;
import br.com.softilux.softiluxapp.data.db.model.representation.StatusChamado;
import br.com.softilux.softiluxapp.data.nosql.IPaperBook;
import br.com.softilux.softiluxapp.data.preferences.Preferences;
import br.com.softilux.softiluxapp.data.repository.interfaces.IChamadoRepository;
import br.com.softilux.softiluxapp.transfer.datatransfer.ChamadoFilter;
import br.com.softilux.softiluxapp.transfer.datatransfer.ChamadoResponse;
import br.com.softilux.softiluxapp.transfer.services.IChamadoService;
import br.com.softilux.softiluxapp.ui.chamado.ChamadoItemRow;
import br.com.softilux.softiluxapp.utils.rx.SchedulerProvider;
import io.reactivex.Observable;
import io.reactivex.disposables.CompositeDisposable;

/**
 * Created by Vinicius on 30,Abril,2019
 */
public class ChamadoRepository extends BaseRepository implements IChamadoRepository {

    private final IChamadoService chamadoService;

    private final ChamadoDAO mChamadoDAO;

    @Inject
    public ChamadoRepository(SchedulerProvider schedulerProvider,
                             CompositeDisposable compositeDisposable,
                             Gson gson, Preferences preferences, IPaperBook iPaperBook,
                             IChamadoService chamadoService, ChamadoDAO chamadoDAO) {
        super(schedulerProvider, compositeDisposable, gson, preferences, iPaperBook);
        this.chamadoService = chamadoService;
        this.mChamadoDAO = chamadoDAO;
    }

    public IChamadoService getChamadoService() {
        return chamadoService;
    }

    @Override
    public Observable<ResponseListTransfer<ChamadoResponse>> loadAllNetworking(final ChamadoFilter chamadoFilter) {
        return getChamadoService().loadAll(chamadoFilter);
    }

    /**
     * Salvar os chamados na base de dados a partir do response
     * da sincronização.
     * */
    @Override
    public Observable<Boolean> observableSaveAllDatabase(List<ChamadoModel> models) {
        return Observable.fromCallable(new Callable<Boolean>() {
            @Override
            public Boolean call() throws Exception {
                for (ChamadoModel entity : models) {
                    if (!(mChamadoDAO.countByIdChamadoWeb(entity.getIdChamadoWeb()) > 0)) {
                        mChamadoDAO.insert(entity);
                    } else {
                        ChamadoModel chamadoModel = mChamadoDAO.findByIdChamadoWeb(entity.getIdChamadoWeb());
                        entity.setIdApp(chamadoModel.getIdApp());
                        mChamadoDAO.update(chamadoModel);
                    }
                }
                return true;
            }
        });
    }

    public void saveAll(List<ChamadoModel> models) {
        for (ChamadoModel persistences : models) {
            if ( !(mChamadoDAO.countByIdChamadoWeb(persistences.getIdChamadoWeb()) > 0)) {
                mChamadoDAO.insert(persistences);
            }
        }
    }

    public List<ChamadoModel> loadAllDatabase() {
        return mChamadoDAO.getAll();
    }

    public Observable<List<ChamadoModel>> observableLoadAllDatabase() {
        return Observable.fromCallable(new Callable<List<ChamadoModel>>() {
            @Override
            public List<ChamadoModel> call() throws Exception {
                return mChamadoDAO.getAll();
            }
        });
    }

    public Observable<List<ChamadoItemRow>> observableLoadAllDatabaseGroup() {
        return Observable.fromCallable(new Callable<List<ChamadoItemRow>>() {
            @Override
            public List<ChamadoItemRow> call() throws Exception {
                List<ChamadoItemRow> chamados = new ArrayList<>();
                List<ChamadoModel> models = new ArrayList<>();

                int countAtivo = mChamadoDAO.countByStatusApp(Collections.singletonList(StatusChamado.ATIVO.getStatus()));

                if (countAtivo > 0) {
                    ChamadoModel ativo = mChamadoDAO.findFirstByStatusApp(StatusChamado.ATIVO.getStatus());
                    models.add(ativo);
                    chamados.add(ChamadoItemRow.criarGrupoChamado(StatusChamado.ATIVO, countAtivo));
                    chamados.addAll(ChamadoItemRow.converterModelParaItem(Collections.singletonList(ativo)));
                }

                int countAtrasados = mChamadoDAO.countByStatusApp(Collections.singletonList(StatusChamado.ATRASADOS.getStatus()));

                if (countAtrasados > 0) {
                    List<ChamadoModel> atrasados = mChamadoDAO.findAllByStatusApp(Collections.singletonList(StatusChamado.ATRASADOS.getStatus()));
                    models.addAll(atrasados);
                    chamados.add(ChamadoItemRow.criarGrupoChamado(StatusChamado.ATRASADOS, countAtrasados));
                    chamados.addAll(ChamadoItemRow.converterModelParaItem(atrasados));
                }

                int countLiberadosAtendimento = mChamadoDAO.countByStatusApp(StatusChamado.getStatusLiberadosParaAtendimento());

                if (countLiberadosAtendimento > 0) {
                    List<ChamadoModel> liberados = mChamadoDAO.findAllByStatusApp(StatusChamado.getStatusLiberadosParaAtendimento());
                    models.addAll(liberados);
                    chamados.add(ChamadoItemRow.criarGrupoChamado(StatusChamado.LIBERADO_PARA_ATENDIMENTO, countLiberadosAtendimento));
                    chamados.addAll(ChamadoItemRow.converterModelParaItem(liberados));
                }

                salvarNomesDosClientes(models);

                return chamados;
            }

        });
    }

    private void salvarNomesDosClientes(List<ChamadoModel> models) {
        if (CollectionUtils.isNotEmpty(models)) {
            Set<String> nomes = models.stream()
                    .map(ChamadoModel::getNmCliente)
                    .collect(Collectors.toSet());

            getPaperBook().salvarNomesClientes(nomes);
        }
    }

    @Override
    public Observable<ChamadoModel> observableFindOneDatabase(final Long id) {
        return Observable.fromCallable(new Callable<ChamadoModel>() {
            @Override
            public ChamadoModel call() throws Exception {
                return mChamadoDAO.findByIdChamadoWeb(id);
            }
        });
    }

    public Observable<List<ChamadoItemRow>> observableLoadAllWithFilterDatabaseGroup(@NonNull ChamadoFilterModel chamadoFilterModel) {
        return Observable.fromCallable(new Callable<List<ChamadoItemRow>>() {
            @Override
            public List<ChamadoItemRow> call() throws Exception {
                List<ChamadoItemRow> chamados = new ArrayList<>();

                QueryBuilder queryBuilder = new QueryBuilder();

                queryBuilder.appendSelect("SELECT * FROM chamados WHERE 1=1 ");
                queryBuilder.appendCountSelect("SELECT COUNT(*) FROM chamados WHERE 1=1 ");

                queryBuilder.append(" AND nm_cliente LIKE ? ", chamadoFilterModel.isNomeClienteInformado(),
                        chamadoFilterModel.getNomeCliente());

                queryBuilder.append(" AND serie_equipamento LIKE ? ", chamadoFilterModel.isSerieEquipamentoInformada(),
                        chamadoFilterModel.getSerieEquipamento());

                queryBuilder.append(" AND seq_os = ? ", chamadoFilterModel.isNumeroChamadoSistemaInformado(),
                        chamadoFilterModel.getNumeroChamadoSistema());

                queryBuilder.append(" AND id_chamado_web = ? ", chamadoFilterModel.isNumeroChamadoWebInformado(),
                        chamadoFilterModel.getNumeroChamadoWeb());


                queryBuilder.append(" AND status_app = ? ", chamadoFilterModel.isStatusChamadoInformado(),
                        chamadoFilterModel.getStatusChamado());

                if (chamadoFilterModel.isStatusChamadoInformado()) {
                    int countByStatus = mChamadoDAO.countAllWithFilter(queryBuilder.getCountResult());
                    if (countByStatus > 0) {
                        List<ChamadoModel> models = mChamadoDAO.getAllWithFilter(queryBuilder.getRawResultList());
                        chamados.add(ChamadoItemRow.criarGrupoChamado(chamadoFilterModel.getStatusChamadoAsEnum(), countByStatus));
                        chamados.addAll(ChamadoItemRow.converterModelParaItem(models));
                    }
                } else {

                    // Ativo
                    QueryBuilder queryBuildAtivos =  queryBuilder.copy();
                    queryBuildAtivos.append(" AND status_app = ? ", true, StatusChamado.ATIVO.getStatus());

                    int countAtivo = mChamadoDAO.countAllWithFilter(queryBuildAtivos.getCountResult());
                    if (countAtivo > 0) {
                        List<ChamadoModel> models = mChamadoDAO.getAllWithFilter(queryBuildAtivos.getRawResultList());
                        chamados.add(ChamadoItemRow.criarGrupoChamado(StatusChamado.ATIVO, countAtivo));
                        chamados.addAll(ChamadoItemRow.converterModelParaItem(models));
                    }

                    // Atrados
                    QueryBuilder queryBuildAtrasados = queryBuilder.copy();
                    queryBuildAtrasados.append(" AND status_app = ? ", true, StatusChamado.ATRASADOS.getStatus());

                    int countAtrasados = mChamadoDAO.countAllWithFilter(queryBuildAtrasados.getCountResult());
                    if (countAtrasados > 0) {
                        List<ChamadoModel> models = mChamadoDAO.getAllWithFilter(queryBuildAtrasados.getRawResultList());
                        chamados.add(ChamadoItemRow.criarGrupoChamado(StatusChamado.ATRASADOS, countAtrasados));
                        chamados.addAll(ChamadoItemRow.converterModelParaItem(models));
                    }
                    // Liberados para atendimento
                    QueryBuilder queryBuildLiberados = queryBuilder.copy();
                    queryBuildLiberados.append(" AND status_app = ? ", true, StatusChamado.LIBERADO_PARA_ATENDIMENTO.getStatus());

                    int countLiberados = mChamadoDAO.countAllWithFilter(queryBuildLiberados.getCountResult());
                    if (countLiberados > 0) {
                        List<ChamadoModel> models = mChamadoDAO.getAllWithFilter(queryBuildLiberados.getRawResultList());
                        chamados.add(ChamadoItemRow.criarGrupoChamado(StatusChamado.LIBERADO_PARA_ATENDIMENTO, countLiberados));
                        chamados.addAll(ChamadoItemRow.converterModelParaItem(models));
                    }
                }
                return chamados;
            }
        });
    }

    @Override
    public Observable<Boolean> observableDeleteAllDatabase() {
        return Observable.fromCallable(new Callable<Boolean>() {
            @Override
            public Boolean call() throws Exception {
                mChamadoDAO.deleteAll();
                return true;
            }
        });
    }

    public Observable<ResponseTransfer<ChamadoResponse>> getAtendimentoAtivoComTimelines(Long chamadoNumero) {
        return getChamadoService().getAtendimentoAtivoComTimelines(chamadoNumero);
    }
}
